<?php

class R2
{
    private $accessKey;
    private $secretKey;
    private $endpoint;
    private $region = 'us-east-1';
    private $service = 's3';

    public function __construct($endpoint, $accessKey, $secretKey)
    {
        $this->endpoint = rtrim($endpoint, '/');
        $this->accessKey = $accessKey;
        $this->secretKey = $secretKey;
    }

    public function listObjects($bucket, $prefix = '')
    {
        $query = [];
        if ($prefix) {
            $query['prefix'] = $prefix;
        }
        $query['list-type'] = '2'; // ListObjectsV2

        $response = $this->request('GET', $bucket, '', $query);
        $xml = simplexml_load_string($response['body']);

        $files = [];
        if (isset($xml->Contents)) {
            foreach ($xml->Contents as $content) {
                $files[] = [
                    'key' => (string) $content->Key,
                    'size' => (int) $content->Size,
                    'last_modified' => (string) $content->LastModified
                ];
            }
        }
        return $files;
    }

    public function putObject($bucket, $key, $filePath)
    {
        if (!file_exists($filePath)) {
            throw new Exception("File not found: $filePath");
        }

        // Use streaming upload
        $fp = fopen($filePath, 'r');
        $size = filesize($filePath);

        try {
            return $this->request('PUT', $bucket, $key, [], null, $fp, $size);
        } finally {
            if (is_resource($fp))
                fclose($fp);
        }
    }

    public function deleteObject($bucket, $key)
    {
        return $this->request('DELETE', $bucket, $key);
    }

    public function getObject($bucket, $key)
    {
        return $this->request('GET', $bucket, $key);
    }

    public function getHead($bucket, $key)
    {
        return $this->request('HEAD', $bucket, $key);
    }

    private function request($method, $bucket, $key = '', $query = [], $payload = null, $fileHandle = null, $fileSize = 0)
    {
        $host = parse_url($this->endpoint, PHP_URL_HOST);

        $uri = '/' . $bucket;
        if ($key) {
            $uri .= '/' . implode('/', array_map('rawurlencode', explode('/', $key)));
        }

        ksort($query);
        $queryString = http_build_query($query, '', '&', PHP_QUERY_RFC3986);
        $fullUrl = $this->endpoint . $uri;
        if ($queryString) {
            $fullUrl .= '?' . $queryString;
        }

        // For large uploads, use UNSIGNED-PAYLOAD to avoid hashing entire file in memory
        $contentHash = ($fileHandle) ? 'UNSIGNED-PAYLOAD' : hash('sha256', (string) $payload);

        $headers = [
            'Host' => $host,
            'x-amz-date' => gmdate('Ymd\THis\Z'),
            'x-amz-content-sha256' => $contentHash
        ];

        $authorization = $this->getSignature($method, $uri, $queryString, $headers, $contentHash);
        $headers['Authorization'] = $authorization;

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $fullUrl);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);

        $httpHeaders = [];
        foreach ($headers as $k => $v) {
            $httpHeaders[] = "$k: $v";
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER, $httpHeaders);

        if ($fileHandle) {
            curl_setopt($ch, CURLOPT_PUT, true);
            curl_setopt($ch, CURLOPT_INFILE, $fileHandle);
            curl_setopt($ch, CURLOPT_INFILESIZE, $fileSize);
            curl_setopt($ch, CURLOPT_UPLOAD, true);
        } elseif ($payload) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        }

        curl_setopt($ch, CURLOPT_HEADER, true);

        $response = curl_exec($ch);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $responseHeaders = substr($response, 0, $headerSize);
        $responseBody = substr($response, $headerSize);

        if ($httpCode >= 400) {
            throw new Exception("R2 Error [$httpCode]: " . $responseBody);
        }

        return [
            'code' => $httpCode,
            'headers' => $responseHeaders,
            'body' => $responseBody
        ];
    }

    private function getSignature($method, $uri, $queryString, $headers, $payloadHash)
    {
        $algorithm = 'AWS4-HMAC-SHA256';
        $amzDate = $headers['x-amz-date'];
        $dateStamp = substr($amzDate, 0, 8);
        $credentialScope = "$dateStamp/{$this->region}/{$this->service}/aws4_request";

        // Canonical Headers
        $canonicalHeaders = '';
        $signedHeaders = [];
        ksort($headers);
        foreach ($headers as $k => $v) {
            $lowerK = strtolower($k);
            $canonicalHeaders .= "$lowerK:" . trim($v) . "\n";
            $signedHeaders[] = $lowerK;
        }
        $signedHeadersString = implode(';', $signedHeaders);

        // Canonical Request
        $canonicalRequest = "$method\n" .
            "$uri\n" .
            "$queryString\n" .
            "$canonicalHeaders\n" .
            "$signedHeadersString\n" .
            $payloadHash;

        // String to Sign
        $stringToSign = "$algorithm\n$amzDate\n$credentialScope\n" . hash('sha256', $canonicalRequest);

        // Signing Key
        $kDate = hash_hmac('sha256', $dateStamp, "AWS4" . $this->secretKey, true);
        $kRegion = hash_hmac('sha256', $this->region, $kDate, true);
        $kService = hash_hmac('sha256', $this->service, $kRegion, true);
        $kSigning = hash_hmac('sha256', 'aws4_request', $kService, true);

        // Signature
        $signature = hash_hmac('sha256', $stringToSign, $kSigning);

        return "$algorithm Credential={$this->accessKey}/$credentialScope, SignedHeaders=$signedHeadersString, Signature=$signature";
    }
}
